"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var chai_1 = require("chai");
require("mocha");
var Cleverbot_1 = require("../Cleverbot");
var User_1 = require("../User");
var testingKey;
if (process.env.KEY)
    testingKey = process.env.KEY;
else
    testingKey = require('../../config.json').key;
var cleverbot;
var multibot;
describe('Invoking constructors', function () {
    it('Correct API key should work as intended', function () {
        chai_1.expect(function () { return cleverbot = new Cleverbot_1.Cleverbot(testingKey ? testingKey : ''); }).to.not.throw(Error);
    });
    it('Correct API key should work as intended on multiUser', function () {
        chai_1.expect(function () { return multibot = new Cleverbot_1.Cleverbot(testingKey ? testingKey : '', true); }).to.not.throw(Error);
    });
});
describe('Sending requests normally without multiUser', function () {
    it('Querying without multiUser.', function (done) {
        cleverbot.say('test').then(function (response) {
            chai_1.expect(response).to.be.a('string');
        }).then(function () {
            done();
        });
    }).timeout(15000);
    it('Message counts should be incrementing without multiUser', function () {
        chai_1.expect(cleverbot.callAmount).to.equal(1);
    });
    it('Fetching single user history', function () {
        chai_1.expect(cleverbot.getHistory()).to.have.lengthOf(1);
    });
    it('Getting api key', function () {
        chai_1.expect(cleverbot.apiKey).to.be.a('string');
    });
});
describe('User Errors without multiUser', function () {
    it('Throwing error on fetching user without multiUser', function () {
        chai_1.expect(function () { return cleverbot.getUser('1'); }).to.throw(Error);
    });
    it('Throwing error on setting user emotion without multiUser', function () {
        chai_1.expect(function () { return cleverbot.setEmotion(0, 1); }).to.throw(Error);
    });
    it('Throwing error on setting user engagement without multiUser', function () {
        chai_1.expect(function () { return cleverbot.setEngagement(0, 1); }).to.throw(Error);
    });
    it('Throwing error on setting user regard without multiUser', function () {
        chai_1.expect(function () { return cleverbot.setRegard(0, 1); }).to.throw(Error);
    });
    it('Throwing error on fetching user mood without multiUser', function () {
        chai_1.expect(function () { return cleverbot.getMood(1); }).to.throw(Error);
    });
    it('Throwing error on getting user data without multiUser', function () {
        chai_1.expect(function () { return cleverbot.getUser(1); }).to.throw(Error);
    });
});
describe('Cleverbot moods without multiUser', function () {
    it('Setting regard out of bounds should throw errors', function () {
        chai_1.expect(function () { return cleverbot.setRegard(101); }).to.throw(RangeError, "Regard must be a value between 0 and 100.");
    });
    it('Setting emotion out of bounds should throw errors', function () {
        chai_1.expect(function () { return cleverbot.setEmotion(101); }).to.throw(RangeError, "Emotion must be a value between 0 and 100.");
    });
    it('Setting engagement out of bounds should throw errors', function () {
        chai_1.expect(function () { return cleverbot.setEngagement(101); }).to.throw(RangeError, "Engagement must be a value between 0 and 100.");
    });
    it('setEngagement should be calling without errors', function () {
        chai_1.expect(function () { return cleverbot.setEngagement(20); }).to.not.throw(RangeError);
    });
    it('setRegard should be calling without errors', function () {
        chai_1.expect(function () { return cleverbot.setRegard(20); }).to.not.throw(RangeError);
    });
    it('setEmotion should be calling without errors', function () {
        chai_1.expect(function () { return cleverbot.setEmotion(20); }).to.not.throw(RangeError);
    });
    it('Cleverbot moods should be setting properly', function () {
        chai_1.expect(cleverbot.getMood()).to.deep.equal({
            emotion: 20,
            engagement: 20,
            regard: 20
        });
    });
});
describe('Sending requests in multiUser', function () {
    var input = 'test';
    var reply;
    it('Querying with multiUser.', function (done) {
        multibot.say('test', '1').then(function (response) {
            chai_1.expect(response).to.be.a('string');
            reply = response;
        }).then(function () {
            return multibot.say('test', 2);
        }).then(function () { return done(); });
    }).timeout(15000);
    it('Message counts should be incrementing in multiUser', function () {
        chai_1.expect(multibot.callAmount).to.equal(2);
    });
    it('Fetching users ', function () {
        chai_1.expect(multibot.users).to.be.an.instanceof(Array).and.to.have.lengthOf(2);
    });
    it('Fetching user string inputted information properly with a number', function () {
        chai_1.expect(multibot.getUser(1)).to.be.instanceof(User_1.User);
    });
    it('getLast for user history', function () {
        chai_1.expect(multibot.getUser(1).getFirst()).to.eql([input, reply]);
    });
    it('getFirst for user history', function () {
        chai_1.expect(multibot.getUser(1).getLast()).to.eql([input, reply]);
    });
    it('Getting api key', function () {
        chai_1.expect(cleverbot.apiKey).to.be.a('string');
    });
});
describe('Cleverbot moods with multiUser', function () {
    it('Setting regard out of bounds should throw errors', function () {
        chai_1.expect(function () { return multibot.setRegard(101); }).to.throw(RangeError, "Regard must be a value between 0 and 100.");
    });
    it('Setting emotion out of bounds should throw errors', function () {
        chai_1.expect(function () { return multibot.setEmotion(101); }).to.throw(RangeError, "Emotion must be a value between 0 and 100.");
    });
    it('Setting engagement out of bounds should throw errors', function () {
        chai_1.expect(function () { return multibot.setEngagement(101); }).to.throw(RangeError, "Engagement must be a value between 0 and 100.");
    });
    it('setEngagement should be calling without errors', function () {
        chai_1.expect(function () { return multibot.setEngagement(20, 1); }).to.not.throw(Error);
    });
    it('setEmotion should be calling without errors', function () {
        chai_1.expect(function () { return multibot.setEmotion(20, 1); }).to.not.throw(Error);
    });
    it('setRegard should be calling without errors', function () {
        chai_1.expect(function () { return multibot.setRegard(20, 1); }).to.not.throw(Error);
    });
    it('Cleverbot moods should be setting properly', function () {
        chai_1.expect(multibot.getMood(1)).to.deep.equal({
            emotion: 20,
            engagement: 20,
            regard: 20
        });
    });
});
describe('User Errors with multiUser', function () {
    it('Throwing error on fetching user without multiUser', function () {
        chai_1.expect(function () { return multibot.getUser('1'); }).to.not.throw(Error);
    });
    it('Throwing error on setting user emotion without multiUser', function () {
        chai_1.expect(function () { return multibot.setEmotion(0); }).to.throw(Error);
    });
    it('Throwing error on setting user engagement without multiUser', function () {
        chai_1.expect(function () { return multibot.setEngagement(0); }).to.throw(Error);
    });
    it('Throwing error on setting user regard without multiUser', function () {
        chai_1.expect(function () { return multibot.setRegard(0); }).to.throw(Error);
    });
    it('Throwing error on getting user data without multiUser', function () {
        chai_1.expect(function () { return multibot.getUser(1); }).to.not.throw(Error);
    });
});
describe('CS with multiUser', function () {
    it('Keeping separate cs states with multiple users', function (done) {
        var csStates = multibot.users.map(function (user) { return user.cs; });
        chai_1.expect(csStates[0]).to.not.equal(csStates[1]);
        done();
    });
});
//# sourceMappingURL=testing.js.map