"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var iconv = require("iconv-lite");
var axios_1 = require("axios");
var Exceptions_1 = require("./Exceptions");
var moment = require("moment");
var util_1 = require("util");
var User_1 = require("./User");
var https = require("https");
var Cleverbot = /** @class */ (function () {
    function Cleverbot(input, multiUser) {
        // the config file is our preset for the mood of subsequent calls
        // when multi user is off
        this.config = {
            apiKey: '',
            mood: {
                emotion: 50,
                engagement: 50,
                regard: 50
            }
        };
        if (typeof input === 'string')
            this.config.apiKey = input;
        else if (typeof input === 'object') {
            this.config.apiKey = input.apiKey;
            if (input.mood) {
                // our default config mood is already 50 for each so we just pass
                input.mood.emotion ? this.setEmotion(input.mood.emotion) : '';
                input.mood.engagement ? this.setEngagement(input.mood.engagement) : '';
                input.mood.regard ? this.setRegard(input.mood.regard) : '';
            }
        }
        else {
            throw new TypeError("Cleverbot constructor expects either a string or an Config object.");
        }
        if (multiUser) {
            this.multiUser = true;
            this._users = new Map();
        }
        else {
            this.multiUser = false;
            this.history = [];
        }
        this.endpoint = 'https://www.cleverbot.com/getreply?key=' + this.config.apiKey;
        this.wrapperName = 'Clevertype';
        this.numberOfAPICalls = 0;
        this.statusCodes = Object.keys(Exceptions_1.Exceptions);
        // the first cs request actually does return us a reply
        this.instance = axios_1.default.create({
            httpsAgent: new https.Agent({
                rejectUnauthorized: false
            })
        });
    }
    Object.defineProperty(Cleverbot.prototype, "encodedWrapperName", {
        get: function () {
            return '&wrapper=' + this.wrapperName;
        },
        enumerable: true,
        configurable: true
    });
    Cleverbot.prototype.encodedEmotion = function (emotion) {
        if (emotion) {
            return '&cb_settings_tweak1=' + emotion;
        }
        else if (!this.config.mood || this.config.mood.emotion === undefined) {
            return '';
        }
        return '&cb_settings_tweak1=' + this.config.mood.emotion;
    };
    Cleverbot.prototype.encodedEngagement = function (engagement) {
        if (engagement) {
            return '&cb_settings_tweak2=' + engagement;
        }
        else if (!this.config.mood || this.config.mood.engagement === undefined) {
            return '';
        }
        return '&cb_settings_tweak2=' + this.config.mood.engagement;
    };
    Cleverbot.prototype.encodedRegard = function (regard) {
        if (regard) {
            return '&cb_settings_tweak3=' + regard;
        }
        else if (!this.config.mood || this.config.mood.regard === undefined) {
            return '';
        }
        return '&cb_settings_tweak3=' + this.config.mood.regard;
    };
    Cleverbot.prototype.encodedCleverbotState = function (state) {
        if (state) {
            return '&cs=' + state;
        }
        if (this.CleverbotState === undefined)
            return '';
        return '&cs=' + this.CleverbotState;
    };
    Cleverbot.encodeInput = function (input) {
        var out = iconv.encode(input, 'utf-8');
        return '&input=' + encodeURIComponent(out.toString());
    };
    Cleverbot.prototype.setCleverbotState = function (state, id) {
        if (this.multiUser && id) {
            var user = this.resolveUser(id);
            user.cs = state;
        }
        else {
            this.CleverbotState = state;
        }
    };
    Cleverbot.prototype.createSingleUserHistory = function (userInput, cleverbotResponse, requestDate) {
        if (this.multiUser || !this.history)
            throw new Error('Tried to create single user history in multiUser mode  ');
        var latestConversation = Math.max.apply(Math, this.history.map(function (hs) { return hs.number; }));
        var history = {};
        // we don't want to just set the two moods
        // equals because that passes it by reference
        // and the mood is something that can change
        history.mood = JSON.parse(JSON.stringify(this.config.mood));
        history.number = !isFinite(latestConversation) ? 1 : latestConversation + 1;
        history.input = userInput;
        history.response = cleverbotResponse;
        history.responseDate = new Date();
        history.requestDate = requestDate;
        history.delay = moment(history.responseDate).diff(history.requestDate);
        history.getConversation = function () { return [history.input, history.response]; };
        return history;
    };
    Cleverbot.prototype.createMultiUserHistory = function (userInput, cleverbotResponse, id, requestDate) {
        var _id;
        typeof id === 'string' ? _id = id : _id = id.toString();
        var user = this.resolveUser(_id, true);
        var latestConversation = Math.max.apply(Math, user.history.map(function (hs) { return hs.number; }));
        var history = {};
        // we don't want to just set the two moods
        // equals because that passes it by reference
        // and the mood is something that can change
        history.mood = JSON.parse(JSON.stringify(user.mood));
        history.number = !isFinite(latestConversation) ? 1 : latestConversation + 1;
        history.input = userInput;
        history.response = cleverbotResponse;
        history.responseDate = new Date();
        history.requestDate = requestDate;
        history.delay = moment(history.responseDate).diff(history.requestDate);
        history.getConversation = function () { return [history.input, history.response]; };
        return history;
    };
    Cleverbot.createUser = function (id, eng, emo, reg) {
        var _id = util_1.isString(id) ? id : id.toString();
        var mood = {
            engagement: eng || 50,
            emotion: emo || 50,
            regard: reg || 50
        };
        return new User_1.User(_id, undefined, mood);
    };
    Cleverbot.prototype.resolveUser = function (user, safe) {
        var id;
        var resolvedUser;
        if (!this.multiUser || this._users === undefined)
            throw new Error("Tried resolving user in non-multi user mode.");
        if (typeof user === 'number') {
            id = user.toString();
        }
        else if (typeof user === 'string') {
            id = user;
        }
        else {
            throw new TypeError("Use must be a string or a number.");
        }
        resolvedUser = this._users.get(id);
        if (resolvedUser === undefined) {
            if (safe) {
                throw new ReferenceError("User " + user + " was not found");
            }
            var engagement = this.config.mood.engagement;
            var emotion = this.config.mood.emotion;
            var regard = this.config.mood.regard;
            resolvedUser = Cleverbot.createUser(id, engagement, emotion, regard);
            this._users.set(id, resolvedUser);
        }
        return resolvedUser;
    };
    Cleverbot.prototype.say = function (message, user) {
        var _this = this;
        var requestDate = new Date();
        var _user;
        if (user) {
            _user = this.resolveUser(user);
        }
        var endpoint = this.endpoint;
        endpoint += this.encodedWrapperName;
        endpoint += Cleverbot.encodeInput(message);
        if (_user) {
            endpoint += this.encodedCleverbotState(_user.cs);
            endpoint += this.encodedEmotion(_user.mood.emotion);
            endpoint += this.encodedEngagement(_user.mood.engagement);
            endpoint += this.encodedRegard(_user.mood.regard);
        }
        else {
            endpoint += this.encodedCleverbotState();
            endpoint += this.encodedEmotion();
            endpoint += this.encodedEngagement();
            endpoint += this.encodedRegard();
        }
        return this.instance.get(endpoint).then(function (res) {
            if (res.statusText && _this.statusCodes.includes(res.statusText.toString())) {
                var errorMessage = Exceptions_1.Exceptions[res.statusText];
                return Promise.reject(errorMessage);
            }
            _this.numberOfAPICalls++;
            var response = res.data.output;
            if (_user) {
                _this.setCleverbotState(res.data.cs, _user.id);
                _user.history.push(_this.createMultiUserHistory(message, response, _user.id, requestDate));
                _user.messages++;
            }
            else if (_this.history && !_this.multiUser) {
                _this.setCleverbotState(res.data.cs);
                _this.history.push(_this.createSingleUserHistory(message, response, requestDate));
            }
            return Promise.resolve(response);
        }).catch(function (err) {
            console.log('Error getting response from cleverbot\n' + err);
            console.log('endpoint: ' + endpoint + '\n');
            return Promise.reject(err);
        });
    };
    Cleverbot.prototype.setEmotion = function (amount, user) {
        if (amount < 0 || amount > 100)
            throw new RangeError("Emotion must be a value between 0 and 100.");
        else if (!user && !this.multiUser)
            this.config.mood.emotion = amount;
        else if (!user && this.multiUser)
            throw new Error("setEmotion requires a user id when it's in multi user mode");
        else if (user && !this.multiUser)
            throw new Error("Can not set emotion without user id when in multi user mode.");
        else if (user) {
            var resolved = this.resolveUser(user, true);
            resolved.mood.emotion = amount;
        }
    };
    Cleverbot.prototype.setEngagement = function (amount, user) {
        if (amount < 0 || amount > 100)
            throw new RangeError("Engagement must be a value between 0 and 100.");
        else if (!user && !this.multiUser)
            this.config.mood.engagement = amount;
        else if (!user && this.multiUser)
            throw new Error("setEngagement requires a user id when it's in multi user mode");
        else if (user && !this.multiUser)
            throw new Error("Can not set engagement without user id when in multi user mode.");
        else if (user) {
            var resolved = this.resolveUser(user, true);
            resolved.mood.engagement = amount;
        }
    };
    Cleverbot.prototype.setRegard = function (amount, user) {
        if (amount < 0 || amount > 100)
            throw new RangeError("Regard must be a value between 0 and 100.");
        else if (!user && !this.multiUser)
            this.config.mood.regard = amount;
        else if (!user && this.multiUser)
            throw new Error("setRegard requires a user id when it's in multi user mode");
        else if (user && !this.multiUser)
            throw new Error("Can not set regard without user id when in multi user mode.");
        else if (user) {
            var resolved = this.resolveUser(user, true);
            resolved.mood.regard = amount;
        }
    };
    Object.defineProperty(Cleverbot.prototype, "callAmount", {
        get: function () {
            return this.numberOfAPICalls;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Cleverbot.prototype, "mood", {
        // for backwards compatibility
        /**
         * @deprecated deprecated method since 2.0.0, use getMood() instead
         */
        get: function () {
            console.warn("mood is deprecated, use getMood() instead.");
            return this.config.mood;
        },
        enumerable: true,
        configurable: true
    });
    Cleverbot.prototype.getMood = function (user) {
        if (!user && !this.multiUser)
            return this.config.mood;
        else if (user && !this.multiUser)
            throw new Error("Can not fetch user mood when not in multiUser mood");
        else if (user && this.multiUser)
            return this.resolveUser(user, true).mood;
        else
            throw new Error("A user id is required when fetching mood in multiUser mode");
    };
    Object.defineProperty(Cleverbot.prototype, "apiKey", {
        get: function () {
            return this.config.apiKey;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Cleverbot.prototype, "users", {
        get: function () {
            if (!this.multiUser || !this._users)
                throw new Error("Tried to fetch users but clevertype is not in multi user mode.");
            return Array.from(this._users.values());
        },
        enumerable: true,
        configurable: true
    });
    Cleverbot.prototype.getUser = function (user) {
        return this.resolveUser(user, true);
    };
    Cleverbot.prototype.getHistory = function (user) {
        if (user)
            return this.resolveUser(user, true).history;
        else
            return this.history;
    };
    return Cleverbot;
}());
exports.Cleverbot = Cleverbot;
//# sourceMappingURL=Cleverbot.js.map